## 	Function to determine parameters of prior distribution of lambdaC linking pCR to pC
##	Function inputs:	q1 = answer to elicitation question asking for P(pCR > pc)
##						q2 = expert's opinion for P(pCR > pC - c1)
##						c1 = non-inferiority margin of the trial
##						a, b = parameters of prior distribution of pC representing what is known about pC without the MYCYC data 
## Function returns: parameters of prior distribution of lambdaC ~ N(alpha, gamma2) (alpha = mean, gamma2 = variance)

D2prior_lambdaC <- function(q1, q2, c1, a, b){
	alpha_gamma = qnorm(q1, mean=0, sd=1, lower.tail=TRUE)
   	fval = vector(mode="numeric", length=2)
   	l1 = as.double(0.01)
   	u1 = as.double(5)
   	fval[1] =  D2lambdaC_sd(l1, alpha_gamma, a, b, q2, c1)
   	fval[2] =  D2lambdaC_sd(u1, alpha_gamma, a, b, q2, c1)
   	if(identical(sign(fval[1]), sign(fval[2]))){
    	stop("Error in answers to Day 2 elicitation questions Q1 and Q2: cannot determine a Normal prior distribution linking CYC 6-month remission rates in MYCYC and MYPAN trial populations.")
    }
    
   	z = uniroot(D2lambdaC_sd, interval=c(l1, u1), alpha_gamma, a, b, q2, c1,lower = l1, upper=u1, f.lower=fval[1], f.upper=fval[2])
   	param = vector(mode="numeric", length=2)
   	## lambda ~ normal(alpha, gamma^2), where param[1] = mu, param[2] = gamma^2
   	param[1] = alpha_gamma/z$root
   	param[2] = (1/z$root)^2
   
   	return(param)
}

## Function to search for value of 1/gamma satisfying P{pCR > pC -c1} = q2.
## Function inputs:	gamma_inv = 1/gamma where gamma is prior sd of lambdaC
##					alpha_gamma = alpha/gamma, where alpha, gamma are prior mean and sd of lambdaC
##					a,b = parameters of prior distribution of pC~Beta(a, b) (representing knowledge of pC without related data)
##					q2 = expert's opinion for  P(pCR > pC -c1) = q2 we wish to target.
##					c2 = non-inferiority margin for the trial
## function returns: P(pCR - pC > -c1; 1/gamma) - q2

D2lambdaC_sd <- function(gamma_inv, alpha_gamma, a, b, q2, c1){

    gridc = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gridc = append(c(0.00001, midp1), gridc)
    gridc = append(gridc, c(midp2, 0.99999))  
    lc = length(gridc)

    wc  = vector(mode="numeric", length=lc)
    wc[1] = (gridc[3]-gridc[1])/6.0
    wc[lc] = (gridc[lc] - gridc[lc-2])/6.0
    for(i in seq(2, lc-1, by=2)){
        wc[i] = 4*(gridc[i+1] - gridc[i-1])/6.0
    }

    for(i in seq(3, lc-2, by=2)){
        wc[i] = (gridc[i+2] - gridc[i-2])/6.0
    } 
        
    int = vector(mode = "numeric", length = lc)
    int1 = as.double(0)
    for(i in 1:lc){
    	## for each mesh point for pc, integrate joint prior density of (pC, pCR) (based on expert opinion alone) over the interval [max{0, pc - c1}, 1]
   		low = max(0, gridc[i] - c1)
   		if(low <= 0){
   			int[i] = dbeta(gridc[i], shape1=a, shape2=b, ncp=0, log=FALSE)
   		}else{		
   			## create a mesh for pCR over the interval [upp, 1] (which always has >= 3 points)
   			m = floor((0.999 - low)/0.002)
   			u1 = low + m*0.002
         	midp1 = (u1 + 0.99999)/2.0 	
     		gridca = as.double(append(seq(low, u1, by = 0.001), append(c(midp1), 0.99999) ))
			lca = length(gridca)

  			wca  = vector(mode="numeric", length=lca)
  			wca[1] = (gridca[3]-gridca[1])/6.0
  			wca[lca] = (gridca[lca] - gridca[lca-2])/6.0
 			for(j in seq(2, lca-1, by=2)){
     			wca[j] = 4*(gridca[j+1] - gridca[j-1])/6.0
			}
 			for(j in seq(3, (lca-2), by=2)){
  				wca[j] = (gridca[j+2] - gridca[j-2])/6.0
 		 	}
 		 	
    		## Evaluating joint density for (pc, pCR)
    		dens =  vector(mode="numeric", length=lca)
    		dens1 = vector(mode="numeric", length=lca) 
    		     
    		dens1 = -0.5*((gamma_inv*log(gridca*(1-gridc[i])/(gridc[i]*(1-gridca))) - alpha_gamma)^2)
    		dens1 = exp(dens1)*gamma_inv/sqrt(2*pi)
    		dens = (gridc[i]^(a-1))*((1-gridc[i])^(b-1))/(beta(a,b)*gridca*(1-gridca))
    		dens1 = dens1*dens
    		int[i] = sum(wca*dens1)
     	}
    }
    int1 = sum(wc*int)
    return(int1 - q2)
}

## 	Function to determine parameters of prior distribution of lambda linking pER to pE
##	Function inputs:	q3 = answer to elicitation question asking for P(pER > pE)
##						q4 = answer to elicitation question asking for P(pER > pE - c1)
##						c1 = non-inferiority margin for the trial
##						d1pEprior = Day 1 pE prior density reflecting what is known without MYCYC data (generated by the Plot_nomycyc function).
## Function returns: alpha, gamma2, parameters of prior distribution of lambdaE ~ N(alpha, gamma2) (alpha = mean, gamma2=variance)

D2prior_lambdaE <- function(q3, q4, c1, d1pEprior){
	alpha_gamma = qnorm(q3, mean=0, sd=1, lower.tail=TRUE)
   	fval = vector(mode="numeric", length=2)
   	l1 = as.double(0.01)
   	u1 = as.double(5)
   	fval[1] =  D2lambdaE_sd(l1, alpha_gamma, d1pEprior, q4, c1)
   	fval[2] =  D2lambdaE_sd(u1, alpha_gamma, d1pEprior, q4, c1)
   	if(identical(sign(fval[1]), sign(fval[2]))){
    	stop("Error in answers to elicitation questions Q3 and Q4: cannot determine a Normal prior distribution linking MMF 6-month remission rates in MYCYC and MYPAN trial populations.")
    }
    
   	z = uniroot(D2lambdaE_sd, interval=c(l1, u1), alpha_gamma, d1pEprior, q4, c1, lower = l1, upper=u1, f.lower=fval[1], f.upper=fval[2])
   	param = vector(mode="numeric", length=2)
   	## lambdaE ~ normal(alpha, gamma^2), where param[1] = mu, param[2] = gamma^2
   	param[1] = alpha_gamma/z$root
   	param[2] = (1/z$root)^2
   
   	return(param)
}

## Function to search for value of 1/gamma satisfying P{pER - pE > -c2} = q4.
## Function inputs:	gamma_inv = 1/gamma where gamma is prior sd of lambdaE
##					alpha_gamma = alpha/gamma, where alpha, gamma are prior mean and sd of lambdaE
##					d1pEprior = contains marginal prior density of pE (not incorporating MYCYC data) evaulated by Plot_nomycyc function
##					q4 = expert's opinion for P(pER -pE > -c1) 
##					c1 = non-inferiority margin for the trial
## function returns: P(pER - pE > -c1; 1/gamma) - q4

D2lambdaE_sd <- function(gamma_inv, alpha_gamma, d1pEprior, q4, c1){

    gride = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gride = append(c(0.00001, midp1), gride)
    gride = append(gride, c(midp2, 0.99999))  
    le = length(gride)

    we  = vector(mode="numeric", length=le)
    we[1] = (gride[3]-gride[1])/6.0
    we[le] = (gride[le] - gride[le-2])/6.0
    for(i in seq(2, le-1, by=2)){
        we[i] = 4*(gride[i+1] - gride[i-1])/6.0
    }
    for(i in seq(3, le-2, by=2)){
        we[i] = (gride[i+2] - gride[i-2])/6.0
    } 
        
    int = vector(mode = "numeric", length = le)
    int1 = as.double(0)
    for(i in 1:le){
    	## for each mesh point for pe, integrate joint prior density for (pE, pER) (based on expert opinion alone) over the interval [max{0, pe - c1}, 1]
   		low = max(0, gride[i] - c1)
   		if(low <= 0){
   			## set int[i] = marginal prior density for pE (not incorporating MYCYC data) evaluated at gride[i] (why test d1pEprior$gride[i] == gride[i])
   			int[i] = d1pEprior$dens[i]
   			if(gride[i] != d1pEprior$gride[i]){
   				stop("grids for pE don't match")
   			}
   		}else{		
   			## create a mesh for pER over the interval [upp, 1] (which always has >= 3 points)
   			m = floor((0.999 - low)/0.002)
   			u1 = low + m*0.002
         	midp1 = (u1 + 0.99999)/2.0 	
     		gridea = as.double(append(seq(low, u1, by = 0.001), append(c(midp1), 0.99999) ))
			lea = length(gridea)

  			wea  = vector(mode="numeric", length=lea)
  			wea[1] = (gridea[3]-gridea[1])/6.0
  			wea[lea] = (gridea[lea] - gridea[lea-2])/6.0
 			for(j in seq(2, lea-1, by=2)){
     			wea[j] = 4*(gridea[j+1] - gridea[j-1])/6.0
			}
 			for(j in seq(3, (lea-2), by=2)){
  				wea[j] = (gridea[j+2] - gridea[j-2])/6.0
 		 	}
    		## Evaluating joint prior density for (pE, pER)
    		dens =  vector(mode="numeric", length=lea)
    		dens1 = vector(mode="numeric", length=lea) 
    		     
    		dens1 = -0.5*((gamma_inv*log(gridea*(1-gride[i])/(gride[i]*(1-gridea))) - alpha_gamma)^2)
    		dens1 = exp(dens1)*gamma_inv/sqrt(2*pi)
    		dens = dens1/(gridea*(1-gridea))
    		dens1 = dens*d1pEprior$dens[i]
    		int[i] = sum(wea*dens1)
     	}
    }
    int1 = sum(we*int)
    return(int1 - q4)
}

## Function evaluating the prior distribution of theta incorporating the MYCYC data
## Function inputs: 	mu, sigma2 = mean and variance of theta~N(mu, sigma2) without knowledge of the MYCYC data
##						mycyc = vector containing the mycyc data
##						a, b = parameters of prior distribution of pC~Beta(a,b) without knowledge of the MYCYC data
##						lCparam, lEparam = vectors containing parameters of prior distributions of lambdaC and lambdaE (means and variances)
##						normEC = normalising constant of joint prior density of (pE, pC)|zr (zr = mycyc data) 
## Function returns: dataframe containing expectation, mode, SD of prior distribution of theta incorporating the MYCYC data.			
D2prior_theta <- function(mu, sigma2, mycyc, a, b, lCparam, lEparam, normEC){
	
	## Create a grid for integrating over theta
	r = as.integer(16)
   	mesh = as.integer(6*r -1)
   	mesh1 = as.integer(2*mesh-1)
   	grid1 = vector(mode="numeric", length= mesh)
   	gridt = vector(mode="numeric", length= mesh1)
   	
   	for(i in 1:mesh){
   		if(i <= (r-1)){
   			grid1[i] = mu + sqrt(sigma2)*(-3-4*log(r/i))
   		}else if((i >= r) & (i<= 5*r)){
   			grid1[i] = mu + sqrt(sigma2)*(-3 + 3*(i-r)/(2*r))
   		}else{
   			grid1[i] = mu + sqrt(sigma2)*(3+ 4*log(r/(6*r -i)))
   		}
   	}   
	## calculating mesh mid-points 
   	for(i in seq(1, mesh, by=1)){
   		gridt[2*i-1] = grid1[i]	
   	}
   	for(i in seq(2, mesh1-1, by=2)){
   		gridt[i] = (gridt[i+1] + gridt[i-1])/2.0
   	}    
	## calculating Simpson's integration weights
   	wtheta  = vector(mode="numeric", length=mesh1)
    wtheta[1] = (gridt[3]-gridt[1])/6.0
  	wtheta[mesh1] = (gridt[mesh1] - gridt[mesh1-2])/6.0
  	for(i in seq(2, (mesh1-1), by=2)){
    	wtheta[i] = 4*(gridt[i+1] - gridt[i-1])/6.0
   	}
   	for(i in seq(3, (mesh1-2), by=2)){
		wtheta[i] = (gridt[i+2] - gridt[i-2])/6.0
   	}
	
	## Create a grid for integrating pC
	gridc = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gridc = append(c(0.00001, midp1), gridc)
    gridc = append(gridc, c(midp2, 0.99999))  
    lc = length(gridc)
	wc  = vector(mode="numeric", length=lc)
  	wc[1] = (gridc[3]-gridc[1])/6.0
  	wc[lc] = (gridc[lc] - gridc[lc-2])/6.0
  	for(i in seq(2, lc-1, by=2)){
    	wc[i] = 4*(gridc[i+1] - gridc[i-1])/6.0
  	}
  	for(i in seq(3, lc-2, by=2)){
    	wc[i] = (gridc[i+2] - gridc[i-2])/6.0
  	}
  	
  	## Create a grid for integrating over dummy variables pER and pCR (use same grid and weights for integrating over each variable)
  	grider = gridc
  	
  	int = vector(mode="numeric", length=mesh1)
  	denspc = vector(mode="numeric", length=lc)
  	
  	int1 = (grider^(mycyc[3] -1))*((1-grider)^(mycyc[4]-1))
  	int2 = (grider^(mycyc[1]-1))*((1-grider)^(mycyc[2]-1))
  	for(i in 1:mesh1){
  	## Integrate over joint density of (theta, pC)|zr	(zr = mycyc data)
  		for(j in 1:lc){
  			int3 = int1*dnorm( log(grider*(1-gridc[j])/((1-grider)*gridc[j]*exp(gridt[i]))), mean = lEparam[1], sd = sqrt(lEparam[2]))
  			int4 = int2*dnorm(log(grider*(1-gridc[j])/((1-grider)*gridc[j])), mean = lCparam[1], sd = sqrt(lCparam[2]))
  			denspc[j] = sum(wc*int3)*sum(wc*int4)*dbeta(gridc[j], shape1 = a, shape2=b)*dnorm(gridt[i], mean=mu, sd = sqrt(sigma2))*normEC
  		}
  		int[i] = sum(wc*denspc)
	}
		
	## Calculate summaries of prior distribution of theta (incorporating MYCYC data)
   	expect = sum(wtheta*gridt*int)
    sdev = sqrt(sum((gridt^2)*int*wtheta) - (expect^2) )
	modecal = data.frame(gridt, int)
	modecal_sort = modecal[order(int, decreasing = TRUE), ]
	mode1 = modecal_sort$gridt[1] 
    
    ci_low=0
    ci_upp=0	
     
    return(data.frame(expect, mode1, sdev, ci_low, ci_upp))
}



## Function to evaluate the joint prior distribution of (pC, pE) incorporating the mycyc data
## at given values of pE and pC

## Function inputs: (a, b) = parameters of the pC prior Beta(a,b) not incorporating the MYCYC data
##					lCparam, lEparam = vectors containing parameters of prior distributions of lambdaC and lambdaE (means and variances)
##					sigmainv = 1/sd of the prior distribution of theta ~N( , ) without knowledge of MYCYC data
##					musigma = mu/sigma, where mu is the mean, sigma is sd of prior of theta~N( , ) without knowledge of MYCYC data
##					mycyc = vector containing mycyc data
## Function outputs: normEC = normalising constant of joint prior density of (pE, pC)|zr (zr = mycyc data)

D2prior_pcpe <- function(a, b, lCparam, lEparam, sigmainv, musigma, mycyc){
	
	## Set up grids of values for pER and pCR
	gridca = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gridca = append(c(0.00001, midp1), gridca)
    gridca = append(gridca, c(midp2, 0.99999))  
    lca = length(gridca)
    
    wca  = vector(mode="numeric", length=lca)
    wca[1] = (gridca[3]-gridca[1])/6.0
    wca[lca] = (gridca[lca] - gridca[lca-2])/6.0
    for(i in seq(2, lca-1, by=2)){
        wca[i] = 4*(gridca[i+1] - gridca[i-1])/6.0
    }
    for(i in seq(3, lca-2, by=2)){
        wca[i] = (gridca[i+2] - gridca[i-2])/6.0
    }
    
	gridea = gridca  
    wea = wca
    dens = vector(mode="numeric", length=lca)
    
    gridc = gridca
    gride = gridc
    denspc = vector(mode="numeric", length=lca)
    int = vector(mode="numeric", length=lca)
    intpca =vector(mode = "numeric", length=lca)
    for(j in 1:lca){
    	const1 = dnorm(log(gridca*(1-gridc[j])/(gridc[j]*(1-gridca))), mean = lCparam[1], sd = sqrt(lCparam[2]))
    	dens = (gridca^(mycyc[1]-1))*((1 - gridca)^(mycyc[2]-1))*const1
    	intpca[j] = sum(wca*dens)
    }
    for(i in 1:lca){
    	const1 = dnorm(log(gridea*(1-gride[i])/(gride[i]*(1-gridea))), mean = lEparam[1], sd = sqrt(lEparam[2]))
    	dens = (gridea^(mycyc[3] - 1))*((1-gridea)^(mycyc[4]-1))*const1
		intpea = sum(wea*dens)
    	for(j in 1:lca){
    		## Integrating across the joint density of (pE, pC)|zr
    		const1 = dnorm(log(gride[i]*(1-gridc[j])/(gridc[j]*(1 - gride[i]))), mean = musigma/sigmainv, sd = 1/sigmainv)
			denspc[j] = const1*intpca[j]*intpea*dbeta(gridc[j], shape1=a, shape2=b)/(gride[i]*(1-gride[i]))
		}
		int[i] = sum(wca*denspc)
	}
	norm = 1/sum(wca*int)
	return(norm)
}


## Function to calculate prior distribution of pC conditioning on related dataset: pC|zR.
## Function inputs:	a,b = parameters of prior distribution for pC~Beta(a,b) reflecting what is known without the mycyc data.
##					lCparam, lEparam = vectors containing parameters of prior distributions of lambdaC and lambdaE (means and variances)
## 					sigmainv, musigma = 1/sd and mean/sd of theta~N(, ) reflecting what is known without the mycyc data
##					mycyc = related dataset
##					normEC = normalising constant of joint prior density of (pE, pC)|zr (zr = mycyc data)
## Function returns: dataframe containing expectation, mode, SD of prior distribution of pC incorporating the MYCYC data.
    
D2prior_pc <- function(a, b, lCparam, lEparam, sigmainv, musigma, mycyc, normEC){
	
  	gridc = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gridc = append(c(0.00001, midp1), gridc)
    gridc = append(gridc, c(midp2, 0.99999))  
    lc = length(gridc)
       
  	wc  = vector(mode="numeric", length=lc)
  	wc[1] = (gridc[3]-gridc[1])/6.0
  	wc[lc] = (gridc[lc] - gridc[lc-2])/6.0
  	for(i in seq(2, lc-1, by=2)){
    	wc[i] = 4*(gridc[i+1] - gridc[i-1])/6.0
  	}
  	for(i in seq(3, lc-2, by=2)){
    	wc[i] = (gridc[i+2] - gridc[i-2])/6.0
  	}
  	
  	## Evaluate g(pC| mycyc) on a grid of values of pC  	
  	gride = gridc
  	we = wc
  	le = lc
  	
  	gridea = gridc
  	gridca = gridc
  	
  	denspc = vector(mode="numeric", length = le)
  	intpc = vector(mode="numeric", length = lc)
  	intpea = vector(mode="numeric", length = le)
  	
  	for(j in 1:le){
  		const1 = dnorm(log(gridea*(1-gride[j])/(gride[j]*(1-gridea))), mean = lEparam[1], sd = sqrt(lEparam[2]))
    	dens = (gridea^(mycyc[3] - 1))*((1-gridea)^(mycyc[4]-1))*const1
		intpea[j] = sum(wc*dens)
  	}
  	
  	for(i in 1:lc){
  		const1 = dnorm(log(gridca*(1-gridc[i])/(gridc[i]*(1-gridca))), mean=lCparam[1], sd = sqrt(lCparam[2]))
    	dens = (gridca^(mycyc[1]-1))*((1 - gridca)^(mycyc[2]-1))*const1
    	intpca = sum(wc*dens)
  		for(j in 1:le){	
    		const1 = dnorm(log(gride[j]*(1-gridc[i])/(gridc[i]*(1 - gride[j]))), mean = musigma/sigmainv, sd = 1/sigmainv)
    		denspc[j] = normEC*const1*intpca*intpea[j]*dbeta(gridc[i], shape1=a, shape2=b)/(gride[j]*(1-gride[j]))
  		}
  		intpc[i] = sum(we*denspc)
  	}
  	
  	## Checking to see whether marginal prior density for pC (given related dataset) is a U or L shaped function of pC.
   	## Integrate prior pC density (given related data) over the interval [lim1, lim2]
   	## If probability in this interval is less than would be the case under a flat density, we conclude
   	## the density could be U or L shaped. 
   	lim1 = as.double(0.001)
	lim2 = as.double(0.999)
	
   	gridc1 = gridc[which(gridc >= lim1 & gridc <= lim2)]
   	int1 = intpc[which(gridc >= lim1 & gridc <= lim2)]
   	lc1 = length(gridc1)
   	
   	wc1  = vector(mode="numeric", length=lc1)
  	wc1[1] = (gridc1[3]-gridc1[1])/6.0
  	wc1[lc1] = (gridc1[lc1] - gridc1[lc1-2])/6.0
 	for(i in seq(2, (lc1-1), by=2)){
		wc1[i] = 4*(gridc1[i+1] - gridc1[i-1])/6.0
	}
	for(i in seq(3, (lc1-2), by=2)){
    	wc1[i] = (gridc1[i+2] - gridc1[i-2])/6.0
 	}
    istop = sum(wc1*int1)
   	if(istop < (lim2 - lim1)){
		stop("Error evaluating prior density for CYC remission rate: can't guarantee accuracy of numerical integration routines when integrating U shaped function")
   	}else{
 	     ## Calculate summaries of prior distribution of pC (incorporating MYCYC data
   	    expect = sum(wc*gridc*intpc)
        sdev = sqrt(sum((gridc^2)*intpc*wc) - (expect^2) )
	    modecal = data.frame(gridc, intpc)
		modecal_sort = modecal[order(intpc, decreasing = TRUE), ]
		mode1 = modecal_sort$gridc[1]
		
   	 	ci_low=0
   	 	ci_upp=0
       	return(data.frame(expect, mode1, sdev, ci_low, ci_upp))
  	}
} 

## Function to evaluate the marginal prior density of pC (given related data) at a mesh of values gridt
## Function inputs: 	gridt: mesh of values at which to evaluate h(pc|zr)
##						a, b: parameters of prior distribution of pC~Beta(a,b) (without related data)
##						lCparam, lEparam = vectors containing parameters of prior distributions of lambdaC and lambdaE (means and variances)
##						sigmainv, musigma = 1/sd and mean/sd of theta~N(, ) reflecting what is known without the mycyc data
##						mycyc = related dataset
##						normEC = normalising constant of joint prior density of (pE, pC)|zr (zr = mycyc data)
## Function returns:	intpc: marginal prior density of pC (given mycyc data) evaluated at exp(gridt)/(1 + exp(gridt)) and multiplied by exp(gridt)/(1 + exp(gridt))

D2pc_odds <- function(gridt, a, b, lCparam, lEparam, sigmainv, musigma, mycyc, normEC){
    
  	gride = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gride = append(c(0.00001, midp1), gride)
    gride = append(gride, c(midp2, 0.99999))  
    le = length(gride)
       
  	we  = vector(mode="numeric", length=le)
  	we[1] = (gride[3]-gride[1])/6.0
  	we[le] = (gride[le] - gride[le-2])/6.0
  	for(i in seq(2, le-1, by=2)){
    	we[i] = 4*(gride[i+1] - gride[i-1])/6.0
  	}
  	for(i in seq(3, le-2, by=2)){
    	we[i] = (gride[i+2] - gride[i-2])/6.0
  	}
  	
  	## Evaluate the prior density of pC|zr at the values contained in pcmesh
  	## and multiply these density values by exp(gridt)/((1 + exp(gridt))^2)
  	pcmesh = exp(gridt)/(1 + exp(gridt))
  	lc = length(pcmesh)
 	gridea = gride
 	gridca = gride
 
 	denspc = vector(mode="numeric", length = le)
  	intpc = vector(mode="numeric", length = lc)
  	intpea = vector(mode="numeric", length = le)
  	
  	for(j in 1:le){
  		const1 = dnorm(log(gridea*(1-gride[j])/(gride[j]*(1-gridea))), mean = lEparam[1], sd = sqrt(lEparam[2]))
    	dens = (gridea^(mycyc[3] - 1))*((1-gridea)^(mycyc[4]-1))*const1
		intpea[j] = sum(we*dens)
  	}
  	
  	for(i in 1:lc){
  		const1 = dnorm(log(gridca*(1-pcmesh[i])/(pcmesh[i]*(1-gridca))), mean=lCparam[1], sd = sqrt(lCparam[2]))
    	dens = (gridca^(mycyc[1]-1))*((1 - gridca)^(mycyc[2]-1))*const1
    	intpca = sum(we*dens)
  		for(j in 1:le){	
    		const1 = dnorm(log(gride[j]*(1-pcmesh[i])/(pcmesh[i]*(1 - gride[j]))), mean = musigma/sigmainv, sd = 1/sigmainv)
    		denspc[j] = normEC*const1*intpca*intpea[j]*dbeta(pcmesh[i], shape1=a, shape2=b)/(gride[j]*(1-gride[j]))
  		}
  		intpc[i] = sum(we*denspc)*exp(gridt[i])/((1 + exp(gridt[i]))^2)
  	}
  	
  	return(intpc)
} 
   

## Function to calculate prior distribution of pE incorporating mycyc data.
## Function inputs:	a,b = parameters of prior distribution for pC~Beta(a, b) reflecting what is known without the related data.
##						lCparam, lEparam = vectors containing parameters of prior distributions of lambdaC and lambdaE (means and variances)
##						sigmainv, musigma = 1/sd and mean/sd of theta~N(, ) reflecting what is known without the mycyc data
##						mycyc = related dataset
##						normEC = normalising constant of joint prior density of (pE, pC)|zr (zr = mycyc data)
## Function returns: dataframe containing expectation, mode, SD of prior distribution of pE incorporating the MYCYC data.

D2prior_pe <- function(a, b, lCparam, lEparam, sigmainv, musigma, mycyc, normEC){
	
  	gride = seq(0.001, 0.999, by=0.001)
    midp1 = (0.00001 + 0.001)*0.5
    midp2 = (0.99999 + 0.999)*0.5
    gride = append(c(0.00001, midp1), gride)
    gride = append(gride, c(midp2, 0.99999))
  	le = length(gride)  	
  	
  	we  = vector(mode="numeric", length=le)
  	we[1] = (gride[3]-gride[1])/6.0
  	we[le] = (gride[le] - gride[le-2])/6.0
 	  for(i in seq(2, (le-1), by=2)){
    	we[i] = 4*(gride[i+1] - gride[i-1])/6.0
 	  }
 	  for(i in seq(3, (le-2), by=2)){
    	we[i] = (gride[i+2] - gride[i-2])/6.0
 	  }
  	
  	gridc = gride
  	gridca = gride
  	gridea = gride
  	
  	wc = we
  	lc = le
  	
  	denspc = vector(mode="numeric", length = lc)
  	intpe = vector(mode="numeric", length = le)
  	intpca = vector(mode="numeric", length = lc)
  	
  	for(j in 1:lc){
  		const1 = dnorm(log(gridca*(1-gridc[j])/(gridc[j]*(1-gridca))), mean = lCparam[1], sd = sqrt(lCparam[2]))
    	dens = (gridca^(mycyc[1]-1))*((1 - gridca)^(mycyc[2]-1))*const1
    	intpca[j] = sum(we*dens)
  	}

  	for(i in 1:le){
  		const1 = dnorm(log(gridea*(1-gride[i])/(gride[i]*(1-gridea))), mean = lEparam[1], sd = sqrt(lEparam[2]))
    	dens = (gridea^(mycyc[3] - 1))*((1-gridea)^(mycyc[4]-1))*const1
		intpea = sum(we*dens)
  		for(j in 1:lc){	
    		const1 = dnorm(log(gride[i]*(1-gridc[j])/(gridc[j]*(1 - gride[i]))), mean = musigma/sigmainv, sd = 1/sigmainv)
    		denspc[j] = normEC*const1*intpca[j]*intpea*dbeta(gridc[j], shape1=a, shape2=b)/(gride[i]*(1-gride[i]))
  		}
  		intpe[i] = sum(wc*denspc)
  	}	

  	## Checking to see whether marginal prior density for pE (given related data) is a U or L shaped function of pE.
   	## Integrate prior pE density over the interval [lim1, lim2]
   	## If probability in this interval is less than would be the case under a flat density, we conclude
   	## the density could be U or L shaped.
   	lim1 = as.double(0.001)
	lim2 = as.double(0.999)
	 
   	gride1 = gride[which(gride >= lim1 & gride <= lim2)]
   	int1 = intpe[which(gride >= lim1 & gride <= lim2)]
   	le1 = length(gride1)
   	we1  = vector(mode="numeric", length=le1)
  	we1[1] = (gride1[3]-gride1[1])/6.0
  	we1[le1] = (gride1[le1] - gride1[le1-2])/6.0
 	for(i in seq(2, (le1-1), by=2)){
		we1[i] = 4*(gride1[i+1] - gride1[i-1])/6.0
	}
	for(i in seq(3, (le1-2), by=2)){
    	we1[i] = (gride1[i+2] - gride1[i-2])/6.0
 	}
    istop = sum(we1*int1)
   	if(istop < (lim2 - lim1)){
      	stop("Error evaluating prior density for MMF remission rate: can't guarantee accuracy of numerical integration routines when integrating U shaped function")
   	}else{
   	  	## Calculate summaries of prior distribution of pE
   	  	expect = sum(we*gride*intpe)
	    sdev = sqrt(sum((gride^2)*intpe*we) - (expect^2) )
	    modecal = data.frame(gride, intpe)
		modecal_sort = modecal[order(intpe, decreasing = TRUE), ]
		mode1 = modecal_sort$gride[1]
    	
    	ci_upp=0
    	ci_low=0
  		return(data.frame(expect, mode1, sdev, ci_low, ci_upp))
  	}
}


    
   
    
